;*********************************************************************
;								     *
;             The Count-down timer processing			     *
;								     *
; Author : Seiichi Inoue  :    *
;*********************************************************************

        LIST            P=PIC16F84A
        INCLUDE         P16F84A.inc
        __CONFIG _HS_OSC & _WDT_OFF & _PWRTE_ON & _CP_OFF

;****************  Label Definition  ********************
c10m    equ     h'0c'           ;Counter adr (10th min)
c01m    equ     h'0d'           ;Counter adr ( 1st min)
c10s    equ     h'0e'           ;Counter adr (10th sec)
c01s    equ     h'0f'           ;Counter adr ( 1st sec)

seg7_0  equ     b'01000000'     ;-gfedcba Pattern 0
seg7_1  equ     b'01111001'     ;         Pattern 1
seg7_2  equ     b'00100100'     ;         Pattern 2
seg7_3  equ     b'00110000'     ;         Pattern 3
seg7_4  equ     b'00011001'     ;         Pattern 4
seg7_5  equ     b'00010010'     ;         Pattern 5
seg7_6  equ     b'00000010'     ;         Pattern 6
seg7_7  equ     b'01111000'     ;         Pattern 7
seg7_8  equ     b'00000000'     ;         Pattern 8
seg7_9  equ     b'00010000'     ;         Pattern 9

seg70   equ     h'10'           ;Pattern 0 set adr
seg71   equ     h'11'           ;Pattern 1 set adr
seg72   equ     h'12'           ;Pattern 2 set adr
seg73   equ     h'13'           ;Pattern 3 set adr
seg74   equ     h'14'           ;Pattern 4 set adr
seg75   equ     h'15'           ;Pattern 5 set adr
seg76   equ     h'16'           ;Pattern 6 set adr
seg77   equ     h'17'           ;Pattern 7 set adr
seg78   equ     h'18'           ;Pattern 8 set adr
seg79   equ     h'19'           ;Pattern 9 set adr

tm_cnt  equ     h'1a'           ;Time adjust counter adr
time_f  equ     h'1b'           ;Time out flag adr
bcd_in  equ     h'1c'           ;BCD input area adr

w_save  equ     h'1d'           ;W Register save adr
s_save  equ     h'1e'           ;STATUS Register save adr

cnt500u equ     h'1f'           ;500usec counter adr
cnt1m   equ     h'20'           ;1msec counter adr

ra0     equ     0               ;RA0 of PORTA
ra1     equ     1               ;RA1 of PORTA
ra2     equ     2               ;RA2 of PORTA
ra3     equ     3               ;RA3 of PORTA
ra4     equ     4               ;RA4 of PORTA

rb6     equ     6               ;RB6 of PORTB

;**************  Debugging mode setting  ****************
;For debugging mode, ";" of next line should be removed.
;#define  _debug

;****************  Program Start  ***********************
        org     0               ;Reset Vector
        goto    init
        org     4               ;Interrupt Vector
        goto    int

;****************  Initial Process  *********************
        org     5
init    bsf     STATUS,RP0      ;Change to Bank1
        movlw   b'00010000'     ;RA4:IN,RA3-0:OUT
        movwf   TRISA           ;Set PORTA
        movlw   b'00000111'     ;RBPU/TOCS/PSA=0,PS=111
        movwf   OPTION_REG      ;Set OPTION_REG
        bcf     STATUS,RP0      ;Change to Bank0
        movlw   b'00001000'     ;Set Relay OFF
        movwf   PORTA           ;Set PORTA
        movlw   seg7_0          ;Set 7segment pattern 0
        movwf   seg70           ;Save pattern 0
        movlw   seg7_1          ;Set 7segment pattern 1
        movwf   seg71           ;Save pattern 1
        movlw   seg7_2          ;Set 7segment pattern 2
        movwf   seg72           ;Save pattern 2
        movlw   seg7_3          ;Set 7segment pattern 3
        movwf   seg73           ;Save pattern 3
        movlw   seg7_4          ;Set 7segment pattern 4
        movwf   seg74           ;Save pattern 4
        movlw   seg7_5          ;Set 7segment pattern 5
        movwf   seg75           ;Save pattern 5
        movlw   seg7_6          ;Set 7segment pattern 6
        movwf   seg76           ;Save pattern 6
        movlw   seg7_7          ;Set 7segment pattern 7
        movwf   seg77           ;Save pattern 7
        movlw   seg7_8          ;Set 7segment pattern 8
        movwf   seg78           ;Save pattern 8
        movlw   seg7_9          ;Set 7segment pattern 9
        movwf   seg79           ;Save pattern 9
        clrf    c10s            ;Clear 10th sec counter
        clrf    c01s            ;Clear  1st sec counter
        bsf     time_f,0        ;Set time out flag OFF

;*************  Timer stand-by Process  *****************
stand_by
        bsf     STATUS,RP0      ;Change to Bank1
        movlw   h'ff'           ;RB7-0:IN
        movwf   TRISB           ;Set PORTB to input mode
        bcf     STATUS,RP0      ;Change to Bank0
        bcf     PORTA,ra0       ;RA0 = 0 (10th min BCD)
        bcf     PORTA,ra1       ;RA1 = 0
        bcf     PORTA,ra2       ;RA2 = 0
#ifdef _debug
        movlw   h'fe'           ;(Debugging step)
#else
        call    t1m             ;Wait 1msec
        movf    PORTB,w         ;Read 10th of min BCD
#endif
        xorlw   h'ff'           ;Reverse 0/1
        andlw   h'0f'           ;Pick-up digit
        movwf   c10m            ;Save 10th of the minute
        bsf     PORTA,ra0       ;RA0 = 1 (1st min BCD)
#ifdef _debug
        movlw   h'ff'           ;(Debugging step)
#else
        call    t1m             ;Wait 1msec
        movf    PORTB,w         ;Read 1st of min BCD
#endif
        xorlw   h'ff'           ;Reverse 0/1
        andlw   h'0f'           ;Pick-up digit
        movwf   c01m            ;Save 1st of the minute
        call    led_cont        ;Call LED Countrol sub
        movf    c10m,w          ;Read 10th of minute
        btfss   STATUS,Z        ;10th of min is 0 ?
        goto    sw_check        ;Jump to start sw check
        movf    c01m,w          ;Read 1st of minute
        btfsc   STATUS,Z        ;1st of min is 0 ?
        goto    stand_by        ;Time is 00:00
sw_check
        bsf     STATUS,RP0      ;Change to Bank1
        bsf     TRISB,rb6       ;Set RB6 to input mode
        bcf     STATUS,RP0      ;Change to Bank0
        bcf     PORTA,ra0       ;RA0 = 0 (Start switch)
        bsf     PORTA,ra1       ;RA1 = 1
        bcf     PORTA,ra2       ;RA2 = 0
#ifndef _debug
        call    t1m             ;Wait 1msec
        btfsc   PORTB,rb6       ;Timer start ?
        goto    stand_by        ;No. Stand-by.
#endif

;**************  Timer start Process  *******************
start   
        bcf     PORTA,ra3       ;Set Relay ON
#ifdef _debug
        movlw   d'255'          ;(Debugging step)
#else
        movlw   d'43'           ;Set Hard timer value
#endif
        movwf   TMR0            ;Set TMR0
#ifdef _debug
        movlw   d'2'            ;(Debugging step)
#else
        movlw   d'46'           ;Set Soft count value
#endif
        movwf   tm_cnt          ;Save time adjust
        movlw   h'a0'           ;GIE=1,TOIE=1
        movwf   INTCON          ;Interruption enable
loop
        call    led_cont        ;Call LED Countrol sub
        movf    time_f,w        ;Read time out flag
        btfsc   STATUS,Z        ;Time out ?
        goto    time_out        ;Yes. Jump to time out

#ifndef _debug
        btfsc   PORTA,ra4       ;Stop switch ON ?
#endif
        goto    loop            ;No. Retry

time_out
        clrf    INTCON          ;Interruptin disable
        goto    init            ;Jump to Initial Process

;**************  LED Control Subroutine  ****************
led_cont
        bsf     STATUS,RP0      ;Change to Bank1
        clrf    TRISB           ;Set PORTB to output mode
        bcf     STATUS,RP0      ;Change to Bank0
        bcf     PORTA,ra0       ;RA0 = 0 (7seg 10th min)
        bcf     PORTA,ra1       ;RA1 = 0
        bsf     PORTA,ra2       ;RA2 = 1
        movf    c10m,w          ;Read 10th of min count
        movwf   bcd_in          ;Set count
        call    bcd_7seg        ;LED control
        bsf     PORTA,ra0       ;RA0 = 1 (7seg 1st min)
        movf    c01m,w          ;Read 1st of min count
        movwf   bcd_in          ;Set count
        call    bcd_7seg        ;LED control
        bcf     PORTA,ra0       ;RA0 = 0 (7seg 10th sec)
        bsf     PORTA,ra1       ;RA1 = 1
        movf    c10s,w          ;Read 10th of sec count
        movwf   bcd_in          ;Set count
        call    bcd_7seg        ;LED control
        bsf     PORTA,ra0       ;RA0 = 1 (7seg 1st sec)
        movf    c01s,w          ;Read 1st of sec count
        movwf   bcd_in          ;Set count
        call    bcd_7seg        ;LED control
        return

;*******  Change BCD to 7segment data Subroutine  *******
bcd_7seg
        movlw   seg70           ;Set 7seg pattern head
        addwf   bcd_in,w        ;Pattern table address
        movwf   FSR             ;Set table address
        movf    INDF,w          ;Read 7seg pattern
        movwf   PORTB           ;Set pattern to PORTB
#ifndef _debug
        call    t1m             ;Wait 1msec
#endif
        return

;*************  1msec Timer Subroutine  *****************
t1m     movlw   2               ;(1)       Set loop cnt1
        movwf   cnt1m           ;(1)       Save loop cnt1
tm1lp1  movlw   d'249'          ;(1)*2     Set loop cnt2
        movwf   cnt500u         ;(1)*2     Save loop cnt2
tm1lp2  nop                     ;(1)*249*2 Time adjust
        nop                     ;(1)*249*2 Time adjust
        decfsZ  cnt500u,f       ;(1)*249*2 cnt500u-1=0 ?
        goto    tm1lp2          ;(2)*248*2 No, continue
        decfsZ  cnt1m,f         ;(1)*2     cnt1m-1=0 ?
        goto    tm1lp1          ;(2)       No. Continue
        return                  ;(2)       Yes. Cnt end
                                ;Total 2501*0.4usec=1msec

;************  Begin Interruption Process  **************
int
        movwf   w_save          ;Save W register
        movf    STATUS,w        ;Read STATUS reg
        movwf   s_save          ;Save STATUS reg
        bcf     STATUS,RP0      ;Change to Bank0
        btfsc   INTCON,T0IF     ;Time out interruption ?
        goto    timer_int       ;Jump to Timer process
 
;************  END of Interruption Process **************
int_end
        movf    s_save,w        ;Read saved STATUS reg
        movwf   STATUS          ;Recover STATUS reg
        swapf   w_save,f        ;Read saved W register
        swapf   w_save,w        ;Recover W register
        retfie        
        
;***********  Time-out interruption Process  ************
timer_int
        bcf     INTCON,T0IF     ;Clear timer int flag
#ifdef _debug
        movlw   d'255'          ;(Debugging step)
#else
        movlw   d'43'           ;Set Hard timer value
#endif
        movwf   TMR0            ;Set TMR0
        decfsZ  tm_cnt,f        ;Time over ?
        goto    int_end         ;No. Retry
#ifdef _debug
        movlw   d'2'            ;(Debugging step)
#else
        movlw   d'46'           ;Set Soft count value
#endif
        movwf   tm_cnt          ;Save time adjust

;*************  Timer count-down Process  ***************
        decfsZ  c01s,w          ;1st of sec is 0 from 1 ?
        goto    countdown       ;No. Jump to countdown

;-----------------  Time-out check  ---------------------
        movf    c10s,w          ;Read 10th of sec counter
        btfss   STATUS,Z        ;10th of sec is 0 ?
        goto    countdown       ;Jump to countdown
        movf    c01m,w          ;Read 1st of min counter
        btfss   STATUS,Z        ;1st of min is 0 ?
        goto    countdown       ;Jump to countdown
        movf    c10m,w          ;Read 10th of min counter
        btfss   STATUS,Z        ;10th of min is 0 ?
        goto    countdown       ;Jump to countdown

time_out1
        clrf    time_f          ;Set time-out flag ON
        goto    int_end         ;Return

;----------------  Timer countdown  ---------------------
countdown
        movf    c01s,w          ;Read 1st of sec counter
        btfss   STATUS,Z        ;1st of sec is 0 ?
        goto    cd_c01s         ;Jump to 1st sec cntdown
        movlw   9               ;Set 9
        movwf   c01s            ;Save 9 to 1st of sec
        movf    c10s,w          ;Read 10th of sec counter
        btfss   STATUS,Z        ;10th of sec is 0 ?
        goto    cd_c10s         ;Jump to 10th min cntdown
        movlw   5               ;Set 5
        movwf   c10s            ;Save 5 to 10th of sec
        movf    c01m,w          ;Read 1st of min counter
        btfss   STATUS,Z        ;1st of min is 0 ?
        goto    cd_c01m         ;Jump to 1st min cntdown
        movlw   9               ;Set 9
        movwf   c01m            ;Save 9 to 1st of min
        movf    c10m,w          ;Read 10th of min counter
        btfss   STATUS,Z        ;10th of min is 0 ?
        goto    cd_c10m         ;Jump to 10th min cntdown
        goto    time_out1       ;Counter is 00:00 already

cd_c01s
        decf    c01s,f          ;Count down 1st of sec
        goto    int_end         ;Return

cd_c10s
        decf    c10s,f          ;Count down 10th of sec
        goto    int_end         ;Return

cd_c01m
        decf    c01m,f          ;Count down 1st of min
        goto    int_end         ;Return

cd_c10m
        decf    c10m,f          ;Count down 10th of min
        goto    int_end         ;Return

;********************************************************
;          END of Count-down timer processing
;********************************************************

        end
